##
# __init__.py: Root module for the spellsnake library.
##
# This module is part of the spellsnake package,
# developed under the rereading Project (https://rereading.space).
#
# Copyright 2025, rereading Project. Licensed under the GNU AGPL v3.
##

from collections.abc import Iterator
from contextlib import contextmanager
from io import IOBase
from pathlib import Path

# We ignore type errors in this import, as mypy does not
# read types from extension modules such as Rust modules
# compiled with PyO3.
import spellsnake._spellbook_bind as spellbook  # type: ignore[import-not-found]


@contextmanager
def maybe_open(filelike: IOBase | Path | str) -> Iterator[IOBase]:
    """
    Given either an already opened file-like object or
    a path to a file, yields a filelike object as a
    context manager.
    """
    if isinstance(filelike, IOBase):
        yield filelike
    else:
        with open(filelike) as f:
            yield f


class Dictionary:
    """
    A runtime dictionary of words to be used in spellchecking.
    """

    _backend: spellbook.Dictionary

    def __init__(self, aff: str, dic: str):
        """
        Given the contents of Hunspell-like dictionary files,
        initializes a new runtime dictionary from that data.

        See <https://zverok.space/blog/2021-01-09-spellchecker-2.html>
        for details on the formats used by these files.

        :param aff: The contents of a Hunspell-like affix compression
            file (that is, containing definitions of various prefixes
            and suffixes).
        :param dic: The contents of a Hunspell-like dictionary file
            (that is, containing words, possibly modified with affix
            flags).
        """
        self._backend = spellbook.Dictionary(aff, str)

    @classmethod
    def load(cls, aff: IOBase | Path | str, dic: IOBase | Path | str):
        """
        Given two filelike objects or paths to files containing Hunspell-like
        dictionary data, loads the contents of those files into a new
        runtime dictionary.

        See :meth:`Dictionary.__init__` for more details about the formats
        of each file.
        """
        with maybe_open(aff) as aff_file, maybe_open(dic) as dic_file:
            return cls(aff_file.read(), dic_file.read())

    def check(self, word: str) -> bool:
        """
        Checks whether a given word is in this runtime
        dictionary.
        """
        return self._backend.check(word)

    def suggest(self, word: str) -> list[str]:
        """
        Returns a list of spelling suggestions for a
        possibly misspelled word.
        """
        return self._backend.suggest(word)

    def add_word(self, word: str) -> None:
        """
        Adds a new word to this runtime dictionary.

        Note that this method does not persist the new word;
        reloading the dictionary will not retain the new word.
        """
        self._backend.add_word(word)

    def remove_word(self, word: str) -> None:
        """
        Removes an existing word from this runtime dictionary.

        Note that this method does not modify any persistent
        storage; reloading the dictionary will not cause the
        existing word to be removed again.
        """
        raise NotImplementedError("not yet implemented")
